#include <string.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <gtk/gtk.h>

#define DEFAULT_GNOME_VOICE_XML "GNOME.voice"

// Stream structure
typedef struct {
    char *url;
    char *location;
    double latitude;
    double longitude;
    GtkWidget *play_button;
    GtkWidget *stop_button;
} Stream;

GList *streams = NULL;
GtkWidget *main_box;
GtkWidget *url_entry;

// Function to load streams from an XML file
void load_streams_from_xml(const char *filename) {
    xmlDocPtr doc;
    xmlNodePtr root_element, station_node, location_node;
    char *url, *location;
    double lat, lon;

    doc = xmlReadFile(filename, NULL, 0);
    if (doc == NULL) {
        g_print("Failed to parse the XML file: %s\n", filename);
        return;
    }

    root_element = xmlDocGetRootElement(doc);
    station_node = root_element->children;

    for (; station_node != NULL; station_node = station_node->next) {
        if (station_node->type == XML_ELEMENT_NODE && xmlStrcmp(station_node->name, BAD_CAST "station") == 0) {
            url = (char *)xmlGetProp(station_node, BAD_CAST "uri");
            location_node = station_node->children;
            location = (char *)xmlNodeGetContent(location_node);

            lat = atof((char *)xmlGetProp(location_node, BAD_CAST "lat"));
            lon = atof((char *)xmlGetProp(location_node, BAD_CAST "lon"));

            Stream *new_stream = g_new(Stream, 1);
            new_stream->url = g_strdup(url);
            new_stream->location = g_strdup(location);
            new_stream->latitude = lat;
            new_stream->longitude = lon;

            new_stream->play_button = gtk_button_new_with_label("Play");
            new_stream->stop_button = gtk_button_new_with_label("Stop");

            streams = g_list_append(streams, new_stream);

            GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
            gtk_box_pack_start(GTK_BOX(hbox), new_stream->play_button, FALSE, FALSE, 0);
            gtk_box_pack_start(GTK_BOX(hbox), new_stream->stop_button, FALSE, FALSE, 0);
            gtk_box_pack_start(GTK_BOX(main_box), hbox, FALSE, FALSE, 5);
        }
    }

    xmlFreeDoc(doc);
}

// Function to save streams to an XML file
void save_streams_to_xml(const char *filename) {
    xmlDocPtr doc = xmlNewDoc(BAD_CAST "1.0");
    xmlNodePtr root_node = xmlNewNode(NULL, BAD_CAST "streams");
    xmlDocSetRootElement(doc, root_node);

    for (GList *l = streams; l != NULL; l = l->next) {
        Stream *stream = (Stream *)l->data;
        xmlNodePtr station_node = xmlNewChild(root_node, NULL, BAD_CAST "station", NULL);
        xmlNewProp(station_node, BAD_CAST "uri", BAD_CAST stream->url);

        xmlNodePtr location_node = xmlNewChild(station_node, NULL, BAD_CAST "location", BAD_CAST stream->location);
        xmlNewProp(location_node, BAD_CAST "lat", BAD_CAST g_strdup_printf("%f", stream->latitude));
        xmlNewProp(location_node, BAD_CAST "lon", BAD_CAST g_strdup_printf("%f", stream->longitude));
    }

    if (xmlSaveFormatFileEnc(filename, doc, "UTF-8", 1) == -1) {
        g_print("Failed to save the XML file: %s\n", filename);
    } else {
        g_print("XML file saved successfully: %s\n", filename);
    }

    xmlFreeDoc(doc);
}

// Callback for adding a new stream
void add_stream(GtkButton *button, gpointer user_data) {
    const char *url = gtk_entry_get_text(GTK_ENTRY(url_entry));
    if (strlen(url) == 0) {
        g_print("URL is empty!\n");
        return;
    }

    Stream *new_stream = g_new(Stream, 1);
    new_stream->url = g_strdup(url);
    new_stream->location = g_strdup("Unknown");
    new_stream->latitude = 0.0;
    new_stream->longitude = 0.0;

    new_stream->play_button = gtk_button_new_with_label("Play");
    new_stream->stop_button = gtk_button_new_with_label("Stop");

    streams = g_list_append(streams, new_stream);

    GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
    gtk_box_pack_start(GTK_BOX(hbox), new_stream->play_button, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), new_stream->stop_button, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(main_box), hbox, FALSE, FALSE, 5);

    gtk_widget_show_all(main_box);

    save_streams_to_xml(DEFAULT_GNOME_VOICE_XML);
}

int main(int argc, char *argv[]) {
    gtk_init(&argc, &argv);

    const char *filename = DEFAULT_GNOME_VOICE_XML;
    if (argc > 1) {
        if (strcmp(argv[1], "--filename") == 0 && argc > 2) {
            filename = argv[2];
        } else {
            g_print("Usage: %s [--filename <filename>]\n", argv[0]);
            return 1;
        }
    }

    GtkWidget *window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "GNOME Voice");
    gtk_window_set_default_size(GTK_WINDOW(window), 800, 600);

    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 10);
    gtk_container_add(GTK_CONTAINER(window), vbox);

    GtkWidget *url_label = gtk_label_new("Enter Stream URL:");
    gtk_box_pack_start(GTK_BOX(vbox), url_label, FALSE, FALSE, 0);

    url_entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(vbox), url_entry, FALSE, FALSE, 0);

    GtkWidget *add_button = gtk_button_new_with_label("Add Stream");
    g_signal_connect(add_button, "clicked", G_CALLBACK(add_stream), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), add_button, FALSE, FALSE, 0);

    main_box = gtk_box_new(GTK_ORIENTATION_VERTICAL, 5);
    gtk_box_pack_start(GTK_BOX(vbox), main_box, TRUE, TRUE, 0);

    g_signal_connect(window, "destroy", G_CALLBACK(gtk_main_quit), NULL);
    gtk_widget_show_all(window);

    load_streams_from_xml(filename);

    gtk_main();

    save_streams_to_xml(filename);

    return 0;
}

